#!/usr/bin/env bash
set -euo pipefail

ROOT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
BACKEND_DIR="$ROOT_DIR/app-backend"
OVERLAY_DIR="$ROOT_DIR/overlay"
FRONTEND_DIR="$ROOT_DIR/web-frontend"

echo "== Maharah Hub Installer =="
echo "Root: $ROOT_DIR"

command -v php >/dev/null 2>&1 || { echo "PHP not found. Install PHP 8.2+ first."; exit 1; }
command -v composer >/dev/null 2>&1 || { echo "Composer not found. Install composer first."; exit 1; }

if [ ! -d "$BACKEND_DIR" ]; then
  echo "== Creating Laravel backend in $BACKEND_DIR =="
  composer create-project laravel/laravel "$BACKEND_DIR"
else
  echo "== Laravel backend already exists. Skipping create-project =="
fi

cd "$BACKEND_DIR"

echo "== Installing required packages (Breeze + Sanctum) =="
composer require laravel/breeze laravel/sanctum -n

echo "== Installing Breeze (blade) =="
php artisan breeze:install blade || true

if command -v npm >/dev/null 2>&1; then
  echo "== Building backend assets (npm) =="
  npm install
  npm run build
else
  echo "npm not found. You can build frontend assets later."
fi

echo "== Copying overlay code =="
rsync -a --delete "$OVERLAY_DIR/" "$BACKEND_DIR/"

if [ ! -f ".env" ]; then
  echo "== Creating .env from .env.example =="
  cp .env.example .env
fi

echo "== Generating app key (if missing) =="
php -r "file_exists('.env') || exit(0);" || true
php artisan key:generate --force

echo "== Publishing Sanctum config =="
php artisan vendor:publish --provider="Laravel\Sanctum\SanctumServiceProvider" --force || true

echo "== Running migrations =="
php artisan migrate --force

echo "== Seeding MVP data =="
php artisan db:seed --class="Database\Seeders\MvpSeeder" --force || true

echo ""
echo "== Backend ready =="
echo "Set your DB credentials in: $BACKEND_DIR/.env"
echo "Point subdomain app.maharah-hub.com to: $BACKEND_DIR/public"
echo ""

echo "== Frontend (Next.js) =="
if [ -d "$FRONTEND_DIR" ]; then
  echo "Next.js source is at: $FRONTEND_DIR"
  echo "Run: cd $FRONTEND_DIR && npm install && npm run build && npm run start"
else
  echo "Frontend folder missing: $FRONTEND_DIR"
fi

echo "Done."
